<?php
/**
 * Plugin Name: FreePOS – Point of Sale for WooCommerce
 * Plugin URI:  https://www.cmsws.com/freepos
 * Description: FreePOS is a point-of-sale and inventory management plugin designed to work with WooCommerce and existing payment gateways.
 * Version:     0.2.0
 * Author:      James Lucas
 * Author URI:  https://www.cmsws.com/
 * Update URI:	https://www.cmsws.com/development/freepos/
 * Text Domain: freepos
 * Domain Path: /languages
 *
 * Requires at least: 6.0
 * Requires PHP:      7.4
 *
 * License:     GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

// Adjust this to match the actual plugin folder/file inside wp-content/plugins
define( 'FREEPOS_PLUGIN_SLUG', 'freepos/freepos.php'); 

// Keep this in sync with your header Version
define( 'FREEPOS_VERSION', '0.2.0' );

// URL to your JSON metadata
define( 'FREEPOS_UPDATE_URL', 'https://www.cmsws.com/development/freepos/updates/freepos-info.json' );

// Include helper functions
require_once __DIR__ . '/helper-functions.php';

/**
 * Main FreePOS plugin class.
 */
final class FreePOS {

	/**
	 * Plugin version.
	 *
	 * @var string
	 */
	const VERSION = '0.2.0';

	/**
	 * Single instance of the class.
	 *
	 * @var FreePOS|null
	 */
	private static $instance = null;

	/**
	 * Get the singleton instance.
	 *
	 * @return FreePOS
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * FreePOS constructor.
	 */
	private function __construct() {
		// Define basic constants.
		$this->define_constants();

		// Core hooks.
		add_action( 'plugins_loaded', [ $this, 'on_plugins_loaded' ] );

		// Front-end routing for the POS screen.
		add_action( 'init', [ $this, 'register_rewrites' ] );
		add_filter( 'query_vars', [ $this, 'register_query_vars' ] );
		add_action( 'template_redirect', [ $this, 'handle_pos_route' ] );

		// Settings and admin pages.
		if ( is_admin() ) {
			add_action( 'admin_menu', [ $this, 'register_admin_menu' ] );
			add_action( 'admin_notices', [ $this, 'admin_notices' ] );
			add_action( 'admin_init', [ $this, 'register_settings' ] );
		}

		// When the POS slug option changes, refresh permalinks.
		add_action( 'update_option_freepos_pos_slug', [ $this, 'on_pos_slug_updated' ], 10, 3 );
	}

	/**
	 * Define plugin constants.
	 */
	private function define_constants() {
		if ( ! defined( 'FREEPOS_VERSION' ) ) {
			define( 'FREEPOS_VERSION', self::VERSION );
		}

		if ( ! defined( 'FREEPOS_PLUGIN_FILE' ) ) {
			define( 'FREEPOS_PLUGIN_FILE', __FILE__ );
		}

		if ( ! defined( 'FREEPOS_PLUGIN_DIR' ) ) {
			define( 'FREEPOS_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
		}

		if ( ! defined( 'FREEPOS_PLUGIN_URL' ) ) {
			define( 'FREEPOS_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
		}
	}

	/**
	 * Fired when all plugins are loaded.
	 */
	public function on_plugins_loaded() {
		// Load text domain for translations.
		load_plugin_textdomain(
			'freepos',
			false,
			dirname( plugin_basename( __FILE__ ) ) . '/languages/'
		);
	}

	/* -------------------------------------------------------------------------
	 * Settings
	 * ---------------------------------------------------------------------- */

	/**
	 * Register plugin settings.
	 */
	public function register_settings() {
		register_setting(
			'freepos_settings',
			'freepos_pos_slug',
			[
				'type'              => 'string',
				'sanitize_callback' => [ $this, 'sanitize_pos_slug' ],
				'default'           => 'pos',
			]
		);
	}

	/**
	 * Sanitize the POS slug.
	 *
	 * @param string $value Raw option value.
	 * @return string
	 */
	public function sanitize_pos_slug( $value ) {
		$value = sanitize_title( $value );
		if ( '' === $value ) {
			$value = 'pos';
		}
		return $value;
	}

	/**
	 * Retrieve the current POS slug.
	 *
	 * @return string
	 */
	public function get_pos_slug() {
		$slug = get_option( 'freepos_pos_slug', 'pos' );
		$slug = $this->sanitize_pos_slug( $slug );
		return $slug;
	}

	/**
	 * When the POS slug changes, flush rewrite rules.
	 *
	 * @param mixed  $old_value Old value.
	 * @param mixed  $value     New value.
	 * @param string $option    Option name.
	 */
	public function on_pos_slug_updated( $old_value, $value, $option ) {
		flush_rewrite_rules();
	}

	/* -------------------------------------------------------------------------
	 * Front-end routing for /pos (or custom slug)
	 * ---------------------------------------------------------------------- */

	/**
	 * Register rewrite rules for the POS screen.
	 */
	public function register_rewrites() {
		$slug = $this->get_pos_slug();

		// Example: ^pos/?$ → index.php?freepos_pos=1
		add_rewrite_rule(
			'^' . $slug . '/?$',
			'index.php?freepos_pos=1',
			'top'
		);
	}

	/**
	 * Register custom query vars.
	 *
	 * @param array $vars Existing vars.
	 * @return array
	 */
	public function register_query_vars( $vars ) {
		$vars[] = 'freepos_pos';
		return $vars;
	}

	/**
	 * Handle the POS route and load the POS screen template.
	 */
	public function handle_pos_route() {
		$is_pos = get_query_var( 'freepos_pos' );

		if ( $is_pos ) {
			$template_file = FREEPOS_PLUGIN_DIR . 'templates/pos-screen.php';

			// Make current slug available to the template.
			$pos_slug = $this->get_pos_slug();

			if ( file_exists( $template_file ) ) {
				include $template_file;
			} else {
				// Basic fallback if template is missing.
				get_header();
				echo '<div class="wrap"><h1>' . esc_html__( 'FreePOS', 'freepos' ) . '</h1>';
				echo '<p>' . esc_html__( 'POS screen template not found.', 'freepos' ) . '</p></div>';
				get_footer();
			}
			exit;
		}
	}

	/* -------------------------------------------------------------------------
	 * Admin: menu + page
	 * ---------------------------------------------------------------------- */

	/**
	 * Register the FreePOS admin menu (as a WooCommerce submenu).
	 */
	public function register_admin_menu() {
		// Only show if WooCommerce exists.
		if ( ! class_exists( 'WooCommerce' ) ) {
			return;
		}

		add_submenu_page(
			'woocommerce',
			__( 'FreePOS', 'freepos' ),  // Page title
			__( 'FreePOS', 'freepos' ),  // Menu title
			'manage_woocommerce',        // Capability
			'freepos',                   // Menu slug
			[ $this, 'render_admin_page' ] // Callback
		);
	}

	/**
	 * Render the main FreePOS admin page (controller).
	 * Collects data, then includes the template.
	 */
	public function render_admin_page() {
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_die( esc_html__( 'You do not have permission to access this page.', 'freepos' ) );
		}

		// Data for the view.
		$wc_version  = defined( 'WC_VERSION' ) ? WC_VERSION : __( 'Unknown', 'freepos' );
		$gateways    = [];
		$gateway_err = '';
		$pos_slug    = $this->get_pos_slug();

		if ( function_exists( 'WC' ) && WC()->payment_gateways() ) {
			$gateways = WC()->payment_gateways()->get_available_payment_gateways();
		} else {
			$gateway_err = __( 'Unable to load payment gateways. Is WooCommerce fully initialized?', 'freepos' );
		}

		// Template path.
		$template_file = FREEPOS_PLUGIN_DIR . 'templates/admin-main-page.php';

		if ( file_exists( $template_file ) ) {
			// Variables are in local scope for the template.
			include $template_file;
		} else {
			// Fallback if the template is missing.
			echo '<div class="wrap"><h1>' . esc_html__( 'FreePOS', 'freepos' ) . '</h1>';
			echo '<p>' . esc_html__( 'Admin template file not found.', 'freepos' ) . '</p></div>';
		}
	}

	/**
	 * Show admin notices (e.g. WooCommerce dependency).
	 */
	public function admin_notices() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Check if WooCommerce is active.
		if ( ! class_exists( 'WooCommerce' ) ) {
			?>
			<div class="notice notice-error">
				<p>
					<?php
					echo esc_html__(
						'FreePOS requires WooCommerce to be installed and active. Please install/activate WooCommerce to use FreePOS.',
						'freepos'
					);
					?>
				</p>
			</div>
			<?php
		}
	}

	/* -------------------------------------------------------------------------
	 * Activation / Deactivation
	 * ---------------------------------------------------------------------- */

	/**
	 * Activation hook callback.
	 */
	public static function activate() {
		// WooCommerce presence check.
		if ( ! class_exists( 'WooCommerce' ) ) {
			if ( ! function_exists( 'deactivate_plugins' ) ) {
				require_once ABSPATH . 'wp-admin/includes/plugin.php';
			}

			deactivate_plugins( plugin_basename( __FILE__ ) );

			wp_die(
				esc_html__(
					'FreePOS requires WooCommerce to be installed and active before it can be activated.',
					'freepos'
				),
				esc_html__( 'FreePOS activation error', 'freepos' ),
				[ 'back_link' => true ]
			);
		}

		// Ensure rewrite rules for the default slug are registered then flushed.
		$instance = self::instance();
		$instance->register_rewrites();
		flush_rewrite_rules();
	}

	/**
	 * Deactivation hook callback.
	 */
	public static function deactivate() {
		// Clean up rewrite rules.
		flush_rewrite_rules();
	}
}

/**
 * Initialize the plugin.
 *
 * @return FreePOS
 */
function freepos() {
	return FreePOS::instance();
}

// Kick it off.
freepos();

// Register activation/deactivation hooks.
register_activation_hook( __FILE__, [ 'FreePOS', 'activate' ] );
register_deactivation_hook( __FILE__, [ 'FreePOS', 'deactivate' ] );
