<?php
// Helper + update functions for FreePOS

defined( 'ABSPATH' ) || exit;

/**
 * Fetch remote metadata from the update server.
 */
function freepos_get_remote_metadata() {
    if ( ! defined( 'FREEPOS_UPDATE_URL' ) ) {
        return false;
    }

    $response = wp_remote_get( FREEPOS_UPDATE_URL, [
        'timeout' => 10,
    ] );

    if ( is_wp_error( $response ) ) {
        return false;
    }

    if ( wp_remote_retrieve_response_code( $response ) !== 200 ) {
        return false;
    }

    $body = wp_remote_retrieve_body( $response );
    $data = json_decode( $body );

    if ( ! $data || ! isset( $data->new_version ) ) {
        return false;
    }

    return $data;
}

/**
 * Hook into the plugin update check and tell WP about new versions.
 */
add_filter( 'pre_set_site_transient_update_plugins', 'freepos_check_for_updates' );

function freepos_check_for_updates( $transient ) {
    if ( empty( $transient->checked ) || ! is_object( $transient ) ) {
        return $transient;
    }

    if ( ! defined( 'FREEPOS_PLUGIN_SLUG' ) || ! defined( 'FREEPOS_VERSION' ) ) {
        return $transient;
    }

    // Only proceed if our plugin is present
    if ( ! isset( $transient->checked[ FREEPOS_PLUGIN_SLUG ] ) ) {
        return $transient;
    }

    $remote = freepos_get_remote_metadata();
    if ( ! $remote ) {
        return $transient;
    }

    $remote_version = $remote->new_version ?? $remote->version ?? null;

    if ( $remote_version && version_compare( FREEPOS_VERSION, $remote_version, '<' ) ) {
        $plugin_info = new stdClass();
        $plugin_info->slug        = $remote->slug ?? 'freepos';
        $plugin_info->plugin      = FREEPOS_PLUGIN_SLUG;
        $plugin_info->new_version = $remote_version;
        $plugin_info->url         = $remote->homepage ?? '';
        $plugin_info->package     = $remote->download_url;

        $transient->response[ FREEPOS_PLUGIN_SLUG ] = $plugin_info;
    }

    return $transient;
}

/**
 * Optional: handle the "View details" popup for the plugin.
 */
add_filter( 'plugins_api', 'freepos_plugins_api_handler', 10, 3 );

function freepos_plugins_api_handler( $result, $action, $args ) {
    if ( 'plugin_information' !== $action ) {
        return $result;
    }

    // Slug should match what's in your JSON (`"slug": "freepos"`)
    if ( empty( $args->slug ) || 'freepos' !== $args->slug ) {
        return $result;
    }

    $remote = freepos_get_remote_metadata();
    if ( ! $remote ) {
        return $result;
    }

    $info = new stdClass();
    $info->name          = $remote->name ?? 'FreePOS';
    $info->slug          = $remote->slug ?? 'freepos';
    $info->version       = $remote->new_version ?? $remote->version ?? '';
    $info->author        = $remote->author ?? '';
    $info->requires      = $remote->requires ?? '';
    $info->tested        = $remote->tested ?? '';
    $info->homepage      = $remote->homepage ?? '';
    $info->download_link = $remote->download_url ?? '';
    $info->sections      = [
        'description' => $remote->sections->description ?? '',
        'changelog'   => $remote->sections->changelog ?? '',
    ];

    return $info;
}
