<?php
/**
 * Plugin Name: FreePOS – Point of Sale for WooCommerce
 * Plugin URI:  https://www.cmsws.com/development/freepos
 * Description: FreePOS is a point-of-sale and inventory management plugin designed to work with WooCommerce and existing payment gateways.
 * Version:     0.3.1
 * Author:      James Lucas
 * Author URI:  https://www.cmsws.com/
 * Text Domain: freepos
 * Domain Path: /languages
 * Update URI:  https://www.cmsws.com/development/freepos/updates/freepos-info.json
 *
 * Requires at least: 6.0
 * Requires PHP:      7.4
 *
 * License:     GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

/**
 * Main FreePOS plugin class.
 */
final class FreePOS {

    /**
     * Plugin version.
     *
     * @var string
     */
    const VERSION = '0.3.1';

    /**
     * Single instance of the class.
     *
     * @var FreePOS|null
     */
    private static $instance = null;

    /**
     * Get the singleton instance.
     *
     * @return FreePOS
     */
    public static function instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * FreePOS constructor.
     */
    private function __construct() {
        // Define basic constants.
        $this->define_constants();

        // Core hooks.
        add_action( 'plugins_loaded', array( $this, 'on_plugins_loaded' ) );

        // Front-end routing for the POS screen.
        add_action( 'init', array( $this, 'register_rewrites' ) );
        add_filter( 'query_vars', array( $this, 'register_query_vars' ) );
        add_action( 'template_redirect', array( $this, 'handle_pos_route' ) );

        // Front-end assets for POS UI.
        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_frontend_assets' ) );

        // AJAX handlers for POS product search.
        add_action( 'wp_ajax_freepos_search_products', array( $this, 'ajax_search_products' ) );
        add_action( 'wp_ajax_nopriv_freepos_search_products', array( $this, 'ajax_search_products' ) );

        // Settings and admin pages.
        if ( is_admin() ) {
            add_action( 'admin_menu', array( $this, 'register_admin_menu' ) );
            add_action( 'admin_notices', array( $this, 'admin_notices' ) );
            add_action( 'admin_init', array( $this, 'register_settings' ) );
        }

        // When the POS slug option changes, refresh permalinks.
        add_action( 'update_option_freepos_pos_slug', array( $this, 'on_pos_slug_updated' ), 10, 3 );
    }

    /**
     * Define plugin constants.
     */
    private function define_constants() {
        if ( ! defined( 'FREEPOS_VERSION' ) ) {
            define( 'FREEPOS_VERSION', self::VERSION );
        }

        if ( ! defined( 'FREEPOS_PLUGIN_FILE' ) ) {
            define( 'FREEPOS_PLUGIN_FILE', __FILE__ );
        }

        if ( ! defined( 'FREEPOS_PLUGIN_DIR' ) ) {
            define( 'FREEPOS_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
        }

        if ( ! defined( 'FREEPOS_PLUGIN_URL' ) ) {
            define( 'FREEPOS_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
        }
    }

    /**
     * Fired when all plugins are loaded.
     */
    public function on_plugins_loaded() {
        // Load text domain for translations.
        load_plugin_textdomain(
            'freepos',
            false,
            dirname( plugin_basename( __FILE__ ) ) . '/languages/'
        );
    }

    /* -------------------------------------------------------------------------
     * Settings
     * ---------------------------------------------------------------------- */

    /**
     * Register plugin settings.
     */
    public function register_settings() {
        register_setting(
            'freepos_settings',
            'freepos_pos_slug',
            array(
                'type'              => 'string',
                'sanitize_callback' => array( $this, 'sanitize_pos_slug' ),
                'default'           => 'pos',
            )
        );
    }

    /**
     * Sanitize the POS slug.
     *
     * @param string $value Raw option value.
     * @return string
     */
    public function sanitize_pos_slug( $value ) {
        $value = sanitize_title( $value );
        if ( '' === $value ) {
            $value = 'pos';
        }
        return $value;
    }

    /**
     * Retrieve the current POS slug.
     *
     * @return string
     */
    public function get_pos_slug() {
        $slug = get_option( 'freepos_pos_slug', 'pos' );
        $slug = $this->sanitize_pos_slug( $slug );
        return $slug;
    }

    /**
     * When the POS slug changes, flush rewrite rules.
     *
     * @param mixed  $old_value Old value.
     * @param mixed  $value     New value.
     * @param string $option    Option name.
     */
    public function on_pos_slug_updated( $old_value, $value, $option ) {
        flush_rewrite_rules();
    }

    /* -------------------------------------------------------------------------
     * Front-end routing for /pos (or custom slug)
     * ---------------------------------------------------------------------- */

    /**
     * Register rewrite rules for the POS screen.
     */
    public function register_rewrites() {
        $slug = $this->get_pos_slug();

        // Example: ^pos/?$ → index.php?freepos_pos=1
        add_rewrite_rule(
            '^' . $slug . '/?$',
            'index.php?freepos_pos=1',
            'top'
        );
    }

    /**
     * Register custom query vars.
     *
     * @param array $vars Existing vars.
     * @return array
     */
    public function register_query_vars( $vars ) {
        $vars[] = 'freepos_pos';
        return $vars;
    }

    /**
     * Check if current request is the POS screen.
     *
     * @return bool
     */
    public function is_pos_request() {
        return (bool) get_query_var( 'freepos_pos' );
    }

    /**
     * Handle the POS route and load the POS screen template.
     */
    public function handle_pos_route() {
        if ( ! $this->is_pos_request() ) {
            return;
        }

        // Require login + manage_woocommerce capability for POS.
        if ( ! is_user_logged_in() || ! current_user_can( 'manage_woocommerce' ) ) {
            // Redirect to login page and back here afterward.
            auth_redirect();
            exit;
        }

        $template_file = FREEPOS_PLUGIN_DIR . 'templates/pos-screen.php';

        // Make current slug available to the template.
        $pos_slug = $this->get_pos_slug();

        if ( file_exists( $template_file ) ) {
            include $template_file;
        } else {
            get_header();
            echo '<div class="wrap"><h1>' . esc_html__( 'FreePOS', 'freepos' ) . '</h1>';
            echo '<p>' . esc_html__( 'POS screen template not found.', 'freepos' ) . '</p></div>';
            get_footer();
        }
        exit;
    }

    /* -------------------------------------------------------------------------
     * Front-end assets for POS UI
     * ---------------------------------------------------------------------- */

    /**
     * Enqueue assets on POS screen.
     */
    public function enqueue_frontend_assets() {
        if ( ! $this->is_pos_request() ) {
            return;
        }

        // Optional CSS file (can be empty to start).
        wp_enqueue_style(
            'freepos-pos',
            FREEPOS_PLUGIN_URL . 'assets/css/freepos-pos.css',
            array(),
            self::VERSION
        );

        // Main POS JS.
        wp_enqueue_script(
            'freepos-pos',
            FREEPOS_PLUGIN_URL . 'assets/js/freepos-pos.js',
            array( 'jquery' ),
            self::VERSION,
            true
        );

        $currency_symbol = function_exists( 'get_woocommerce_currency_symbol' )
            ? get_woocommerce_currency_symbol()
            : '$';

        wp_localize_script(
            'freepos-pos',
            'FreePOS_POS',
            array(
                'ajaxUrl'        => admin_url( 'admin-ajax.php' ),
                'nonce'          => wp_create_nonce( 'freepos_pos' ),
                'currencySymbol' => $currency_symbol,
                'searchMinChars' => 2,
                'posSlug'        => $this->get_pos_slug(),
            )
        );
    }

    /* -------------------------------------------------------------------------
     * AJAX: product search for POS
     * ---------------------------------------------------------------------- */

    /**
     * AJAX handler: search products for POS.
     */
    public function ajax_search_products() {
        // Check nonce.
        check_ajax_referer( 'freepos_pos', 'nonce' );

        // Require logged-in user with manage_woocommerce.
        if ( ! is_user_logged_in() || ! current_user_can( 'manage_woocommerce' ) ) {
            wp_send_json_error( array( 'message' => __( 'Unauthorized.', 'freepos' ) ), 403 );
        }

        if ( ! class_exists( 'WooCommerce' ) ) {
            wp_send_json_error( array( 'message' => __( 'WooCommerce not available.', 'freepos' ) ), 500 );
        }

        $term = isset( $_REQUEST['term'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['term'] ) ) : '';

        if ( '' === $term ) {
            wp_send_json_success( array( 'products' => array() ) );
        }

        // Basic product search. We can refine this later (e.g., SKU search).
        $args = array(
            'status' => 'publish',
            'limit'  => 20,
            'return' => 'objects',
        );

        // Use WooCommerce product query.
        if ( function_exists( 'wc_get_products' ) ) {
            $args['search'] = $term;
            $products       = wc_get_products( $args );
        } else {
            $products = array();
        }

        $data = array();

        foreach ( $products as $product ) {
            /** @var WC_Product $product */
            $price = wc_get_price_to_display( $product );
            $data[] = array(
                'id'       => $product->get_id(),
                'name'     => $product->get_name(),
                'sku'      => $product->get_sku(),
                'price'    => $price,
                'price_ht' => wc_price( $price ),
            );
        }

        wp_send_json_success(
            array(
                'products' => $data,
            )
        );
    }

    /* -------------------------------------------------------------------------
     * Admin: menu + page
     * ---------------------------------------------------------------------- */

    /**
     * Register the FreePOS admin menu (as a WooCommerce submenu).
     */
    public function register_admin_menu() {

        if ( ! class_exists( 'WooCommerce' ) ) {
            return;
        }

        //
        // Top-level FreePOS menu
        //
        add_menu_page(
            __( 'FreePOS', 'freepos' ),
            __( 'FreePOS', 'freepos' ),
            'manage_woocommerce',
            'freepos-dashboard',                // <-- IMPORTANT: changed
            array( $this, 'render_dashboard_page' ),
            'dashicons-cart',
            56
        );

        //
        // Dashboard (first submenu)
        //
        add_submenu_page(
            'freepos-dashboard',                // <-- MUST MATCH the top-level slug
            __( 'Dashboard', 'freepos' ),
            __( 'Dashboard', 'freepos' ),
            'manage_woocommerce',
            'freepos-dashboard',                // <-- Same slug as top-level
            array( $this, 'render_dashboard_page' )
        );

        //
        // Orders
        //
        add_submenu_page(
            'freepos-dashboard',
            __( 'Orders', 'freepos' ),
            __( 'Orders', 'freepos' ),
            'manage_woocommerce',
            'freepos-orders',
            array( $this, 'render_orders_page' )
        );

        //
        // Transactions
        //
        add_submenu_page(
            'freepos-dashboard',
            __( 'Transactions', 'freepos' ),
            __( 'Transactions', 'freepos' ),
            'manage_woocommerce',
            'freepos-transactions',
            array( $this, 'render_transactions_page' )
        );

        //
        // Settings
        //
        add_submenu_page(
            'freepos-dashboard',
            __( 'Settings', 'freepos' ),
            __( 'Settings', 'freepos' ),
            'manage_woocommerce',
            'freepos-settings',
            array( $this, 'render_settings_page' )
        );

        //
        // About
        //
        add_submenu_page(
            'freepos-dashboard',
            __( 'About', 'freepos' ),
            __( 'About', 'freepos' ),
            'manage_woocommerce',
            'freepos-about',
            array( $this, 'render_about_page' )
        );

        //
        // Go to POS
        //
        add_submenu_page(
            'freepos-dashboard',
            __( 'Go to POS', 'freepos' ),
            __( 'Go to POS', 'freepos' ),
            'manage_woocommerce',
            'freepos-goto-pos',
            array( $this, 'render_goto_pos_page' )
        );
    }
#    public function register_admin_menu() {
#        // Only show if WooCommerce exists.
#        if ( ! class_exists( 'WooCommerce' ) ) {
#            return;
#        }
#    }

    /**
     * Render the main FreePOS admin page (controller).
     * Collects data, then includes the template.
     */
    public function render_admin_page() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'freepos' ) );
        }

        // Data for the view.
        $wc_version  = defined( 'WC_VERSION' ) ? WC_VERSION : __( 'Unknown', 'freepos' );
        $gateways    = array();
        $gateway_err = '';
        $pos_slug    = $this->get_pos_slug();

        if ( function_exists( 'WC' ) && WC()->payment_gateways() ) {
            $gateways = WC()->payment_gateways()->get_available_payment_gateways();
        } else {
            $gateway_err = __( 'Unable to load payment gateways. Is WooCommerce fully initialized?', 'freepos' );
        }

        // Template path.
        $template_file = FREEPOS_PLUGIN_DIR . 'templates/admin-main-page.php';

        if ( file_exists( $template_file ) ) {
            // Variables are in local scope for the template.
            include $template_file;
        } else {
            // Fallback if the template is missing.
            echo '<div class="wrap"><h1>' . esc_html__( 'FreePOS', 'freepos' ) . '</h1>';
            echo '<p>' . esc_html__( 'Admin template file not found.', 'freepos' ) . '</p></div>';
        }
    }

    /**
     * Dashboard page: overview + gateways.
     */
    public function render_dashboard_page() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'freepos' ) );
        }

        // Data for the view.
        $wc_version  = defined( 'WC_VERSION' ) ? WC_VERSION : __( 'Unknown', 'freepos' );
        $gateways    = array();
        $gateway_err = '';
        $pos_slug    = $this->get_pos_slug();

        if ( function_exists( 'WC' ) && WC()->payment_gateways() ) {
            $gateways = WC()->payment_gateways()->get_available_payment_gateways();
        } else {
            $gateway_err = __( 'Unable to load payment gateways. Is WooCommerce fully initialized?', 'freepos' );
        }

        $template_file = FREEPOS_PLUGIN_DIR . 'templates/admin-dashboard.php';

        if ( file_exists( $template_file ) ) {
            include $template_file;
        } else {
            echo '<div class="wrap"><h1>' . esc_html__( 'FreePOS', 'freepos' ) . '</h1>';
            echo '<p>' . esc_html__( 'Dashboard template file not found.', 'freepos' ) . '</p></div>';
        }
    }

    /**
     * Order page.
     */
    public function render_orders_page() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'freepos' ) );
        }

        $template_file = FREEPOS_PLUGIN_DIR . 'templates/admin-orders.php';

        if ( file_exists( $template_file ) ) {
            include $template_file;
        } else {
            echo '<div class="wrap"><h1>' . esc_html__( 'FreePOS Orders', 'freepos' ) . '</h1>';
            echo '<p>' . esc_html__( 'Orders template not found.', 'freepos' ) . '</p></div>';
        }
    }

    /**
     * Transaction page.
     */
    public function render_transactions_page() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'freepos' ) );
        }

        $template_file = FREEPOS_PLUGIN_DIR . 'templates/admin-transactions.php';

        if ( file_exists( $template_file ) ) {
            include $template_file;
        } else {
            echo '<div class="wrap"><h1>' . esc_html__( 'FreePOS Transactions', 'freepos' ) . '</h1>';
            echo '<p>' . esc_html__( 'Transactions template not found.', 'freepos' ) . '</p></div>';
        }
    }

    /**
     * Settings page: POS slug etc.
     */
    public function render_settings_page() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'freepos' ) );
        }

        $pos_slug = $this->get_pos_slug();

        $template_file = FREEPOS_PLUGIN_DIR . 'templates/admin-settings.php';

        if ( file_exists( $template_file ) ) {
            include $template_file;
        } else {
            echo '<div class="wrap"><h1>' . esc_html__( 'FreePOS Settings', 'freepos' ) . '</h1>';
            echo '<p>' . esc_html__( 'Settings template file not found.', 'freepos' ) . '</p></div>';
        }
    }

    /**
     * About page.
     */
    public function render_about_page() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'freepos' ) );
        }

        $template_file = FREEPOS_PLUGIN_DIR . 'templates/admin-about.php';

        if ( file_exists( $template_file ) ) {
            include $template_file;
        } else {
            echo '<div class="wrap"><h1>' . esc_html__( 'About FreePOS', 'freepos' ) . '</h1>';
            echo '<p>' . esc_html__( 'About template file not found.', 'freepos' ) . '</p></div>';
        }
    }


    /**
     * "Go to POS" page callback – redirects to front-end POS URL.
     */
    public function render_goto_pos_page() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_die( esc_html__( 'You do not have permission to access this page.', 'freepos' ) );
        }

        $pos_url = home_url( '/' . $this->get_pos_slug() . '/' );

        wp_safe_redirect( $pos_url );
        exit;
    }

    /**
     * Show admin notices (e.g. WooCommerce dependency).
     */
    public function admin_notices() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        // Check if WooCommerce is active.
        if ( ! class_exists( 'WooCommerce' ) ) {
            ?>
            <div class="notice notice-error">
                <p>
                    <?php
                    echo esc_html__(
                        'FreePOS requires WooCommerce to be installed and active. Please install/activate WooCommerce to use FreePOS.',
                        'freepos'
                    );
                    ?>
                </p>
            </div>
            <?php
        }
    }

    /* -------------------------------------------------------------------------
     * Activation / Deactivation
     * ---------------------------------------------------------------------- */

    /**
     * Activation hook callback.
     */
    public static function activate() {
        // WooCommerce presence check.
        if ( ! class_exists( 'WooCommerce' ) ) {
            if ( ! function_exists( 'deactivate_plugins' ) ) {
                require_once ABSPATH . 'wp-admin/includes/plugin.php';
            }

            deactivate_plugins( plugin_basename( __FILE__ ) );

            wp_die(
                esc_html__(
                    'FreePOS requires WooCommerce to be installed and active before it can be activated.',
                    'freepos'
                ),
                esc_html__( 'FreePOS activation error', 'freepos' ),
                array( 'back_link' => true )
            );
        }

        // Ensure rewrite rules for the default slug are registered then flushed.
        $instance = self::instance();
        $instance->register_rewrites();
        flush_rewrite_rules();
    }

    /**
     * Deactivation hook callback.
     */
    public static function deactivate() {
        // Clean up rewrite rules.
        flush_rewrite_rules();
    }
}

/**
 * Initialize the plugin.
 *
 * @return FreePOS
 */
function freepos() {
    return FreePOS::instance();
}

// Kick it off.
freepos();

// Register activation/deactivation hooks.
register_activation_hook( __FILE__, array( 'FreePOS', 'activate' ) );
register_deactivation_hook( __FILE__, array( 'FreePOS', 'deactivate' ) );
