(function ($) {
    'use strict';

    // Localized data from PHP.
    const cfg = window.FreePOS_POS || {};
    const ajaxUrl = cfg.ajaxUrl || '';
    const nonce = cfg.nonce || '';
    const currencySymbol = cfg.currencySymbol || '$';
    const searchMinChars = parseInt(cfg.searchMinChars || 2, 10);

    let searchTimer = null;
    let cart = [];

    function formatMoney(value) {
        const num = parseFloat(value) || 0;
        return currencySymbol + num.toFixed(2);
    }

    function findCartItemIndex(id) {
        return cart.findIndex((item) => item.id === id);
    }

    function updateCartTable() {
        const $tbody = $('#freepos-cart-table tbody');
        const $total = $('#freepos-cart-total');
        const $checkoutBtn = $('#freepos-cart-checkout');

        $tbody.empty();

        if (!cart.length) {
            $tbody.append(
                '<tr class="freepos-cart-empty"><td colspan="6">Your cart is empty.</td></tr>'
            );
            $total.text(formatMoney(0));
            $checkoutBtn.prop('disabled', true);
            return;
        }

        let total = 0;

        cart.forEach((item, index) => {
            const lineSubtotal = item.price * item.qty;
            total += lineSubtotal;

            const row = [
                '<tr data-index="' + index + '">',
                '<td>' + item.name + '</td>',
                '<td>' + (item.sku || '') + '</td>',
                '<td><input type="number" min="1" class="freepos-cart-qty" value="' + item.qty + '"/></td>',
                '<td>' + formatMoney(item.price) + '</td>',
                '<td>' + formatMoney(lineSubtotal) + '</td>',
                '<td><button type="button" class="button link-button freepos-cart-remove">\u00d7</button></td>',
                '</tr>',
            ].join('');

            $tbody.append(row);
        });

        $total.text(formatMoney(total));
        $checkoutBtn.prop('disabled', false);
    }

    function addToCart(product) {
        const existingIndex = findCartItemIndex(product.id);

        if (existingIndex !== -1) {
            cart[existingIndex].qty += 1;
        } else {
            cart.push({
                id: product.id,
                name: product.name,
                sku: product.sku,
                price: parseFloat(product.price) || 0,
                qty: 1,
            });
        }

        updateCartTable();
    }

    function renderSearchResults(products) {
        const $tbody = $('#freepos-search-results tbody');
        const $status = $('#freepos-search-status');

        $tbody.empty();

        if (!products.length) {
            $tbody.append(
                '<tr class="freepos-empty-row"><td colspan="5">No products found.</td></tr>'
            );
            $status.text('');
            return;
        }

        products.forEach((product) => {
            const row = [
                '<tr data-id="' + product.id + '">',
                '<td>' + product.id + '</td>',
                '<td>' + product.name + '</td>',
                '<td>' + (product.sku || '') + '</td>',
                '<td>' + formatMoney(product.price) + '</td>',
                '<td><button type="button" class="button freepos-add-to-cart">Add</button></td>',
                '</tr>',
            ].join('');

            $tbody.append(row);
        });

        $status.text(products.length + ' product(s) found.');
    }

    function searchProducts(term) {
        const $status = $('#freepos-search-status');
        const $tbody = $('#freepos-search-results tbody');

        if (!ajaxUrl || !nonce) {
            $status.text('AJAX configuration error.');
            return;
        }

        $status.text('Searching\u2026');
        $tbody.empty().append(
            '<tr class="freepos-empty-row"><td colspan="5">Searching\u2026</td></tr>'
        );

        $.ajax({
            url: ajaxUrl,
            method: 'GET',
            dataType: 'json',
            data: {
                action: 'freepos_search_products',
                nonce: nonce,
                term: term,
            },
        })
            .done((response) => {
                if (!response || !response.success) {
                    const msg = (response && response.data && response.data.message) || 'Search failed.';
                    $status.text(msg);
                    $tbody.empty().append(
                        '<tr class="freepos-empty-row"><td colspan="5">' + msg + '</td></tr>'
                    );
                    return;
                }

                const products = (response.data && response.data.products) || [];
                renderSearchResults(products);
            })
            .fail(() => {
                $status.text('Error performing search.');
                $tbody.empty().append(
                    '<tr class="freepos-empty-row"><td colspan="5">Error performing search.</td></tr>'
                );
            });
    }

    function bindEvents() {
        // Search input debounce.
        $('#freepos-search-input').on('input', function () {
            const term = $(this).val().trim();

            if (searchTimer) {
                clearTimeout(searchTimer);
            }

            if (!term || term.length < searchMinChars) {
                $('#freepos-search-status').text('');
                $('#freepos-search-results tbody')
                    .empty()
                    .append(
                        '<tr class="freepos-empty-row"><td colspan="5">Start typing to search.</td></tr>'
                    );
                return;
            }

            searchTimer = setTimeout(() => {
                searchProducts(term);
            }, 300);
        });

        // Add to cart from search results.
        $('#freepos-search-results').on('click', '.freepos-add-to-cart', function () {
            const $row = $(this).closest('tr');
            const id = parseInt($row.data('id'), 10);

            if (!id) {
                return;
            }

            const name = $row.find('td:nth-child(2)').text();
            const sku = $row.find('td:nth-child(3)').text();
            const priceText = $row.find('td:nth-child(4)').text().replace(currencySymbol, '');
            const price = parseFloat(priceText) || 0;

            addToCart({
                id: id,
                name: name,
                sku: sku,
                price: price,
            });
        });

        // Cart quantity change.
        $('#freepos-cart-table').on('change', '.freepos-cart-qty', function () {
            const $row = $(this).closest('tr');
            const index = parseInt($row.data('index'), 10);
            let qty = parseInt($(this).val(), 10);

            if (isNaN(qty) || qty < 1) {
                qty = 1;
                $(this).val(qty);
            }

            if (cart[index]) {
                cart[index].qty = qty;
                updateCartTable();
            }
        });

        // Remove from cart.
        $('#freepos-cart-table').on('click', '.freepos-cart-remove', function () {
            const $row = $(this).closest('tr');
            const index = parseInt($row.data('index'), 10);

            if (!isNaN(index)) {
                cart.splice(index, 1);
                updateCartTable();
            }
        });

        // Clear cart.
        $('#freepos-cart-clear').on('click', function () {
            if (!cart.length) {
                return;
            }
            if (window.confirm('Clear the cart?')) {
                cart = [];
                updateCartTable();
            }
        });

        // Placeholder checkout button.
        $('#freepos-cart-checkout').on('click', function () {
            if (!cart.length) {
                return;
            }

            // Placeholder: later we will call another AJAX endpoint to create a Woo order.
            alert('Checkout will be implemented in the next step. Cart currently has ' + cart.length + ' item(s).');
        });
    }

    $(function () {
        if (!$('#freepos-search-input').length) {
            // Not on the POS screen.
            return;
        }

        bindEvents();
        updateCartTable();
    });
})(jQuery);
